#include "HardwareProfile.h"
#include "GenericTypeDefs.h"
#include "spi.h"
#include "delay.h"
#include "led.h"
#include "main.h"
#include "diskio.h"
#include "io.h"
#include "main.h"
#include "usb.h"

/* 

	SPI Module Firmware with DMA support for PIC18 devices...
	
	by Mauro Grassi, 2010.
	
*/

#pragma udata spidata

unsigned char 	cardPower;
unsigned char 	SPI18SmallBuffer[1];

#if (USE_PIC18_DMA)

#pragma code usercode

unsigned char* BulkReadPIC18SPI(unsigned char* data, unsigned short n) 
{	
	DMACON2=0;
	DMACON1bits.TXINC=0;	/* do not increment Tx pointer */
	DMACON1bits.RXINC=1;	/* increment Rx pointer */
	DMABCH=(unsigned char)((n-1)>>8);
	DMABCL=(unsigned char)(0xFF & (n-1));
	TXADDRH=((unsigned int)&SPI18SmallBuffer[0])>>8;
	TXADDRL=((unsigned int)&SPI18SmallBuffer[0]);
	RXADDRH=((unsigned int)data)>>8;
	RXADDRL=((unsigned int)data);
	SDCS=0;
	DMACON1bits.DMAEN=1;
	while(DMACON1bits.DMAEN);
	SDCS=1;
	return (unsigned char*)data;
}

unsigned char* BulkWritePIC18SPI(unsigned char* data, unsigned short n) 
{	
	DMACON2=0;
	DMACON1bits.TXINC=1;	/* increment Tx pointer */
	DMACON1bits.RXINC=0;	/* do not increment Rx pointer */
	DMABCH=(unsigned char)((n-1)>>8);
	DMABCL=(unsigned char)(0xFF & (n-1));
	TXADDRH=((unsigned int)data)>>8;
	TXADDRL=((unsigned int)data);
	RXADDRH=((unsigned int)&SPI18SmallBuffer[0])>>8;
	RXADDRL=((unsigned int)&SPI18SmallBuffer[0]);
	SDCS=0;
	DMACON1bits.DMAEN=1;
	while(DMACON1bits.DMAEN);
	SDCS=1;
	return (unsigned char*)data;
}

#endif

#pragma code usercode

void setMemoryCardPower(int onoff)
{
	SDCS=1;
	if(onoff)
	{
		/* on */
		#if(USE_PMDIS)
			PMDIS0bits.ECCP1MD=0;
			PMDIS1bits.TMR2MD=0;
		#endif	
		PR2=0x07;
		SDPOWER_TRIS=0;
		T2CON=0x04;
		CCPTMRS0=0x08;
		CCP1CON=0x0C;
		RPOR13=14;			// 14=CCP1 output (CCP1/P1A)
		setPWMDuty(0x04);
		initLED();
	}
	else
	{
		/* off */
		SDPOWER_TRIS=1;
		T2CON=0;
		CCP1CON=0;
		#if(USE_PMDIS)
			PMDIS0bits.ECCP1MD=1;
			PMDIS1bits.TMR2MD=1;
		#endif
		initLED();
		LED1_TRIS=1;	
	}
}

void setPWMDuty(unsigned int x)
{
	x=x & 0x3FF;
	if(x & 0x200)CCP1CONbits.DC1B1=1; else CCP1CONbits.DC1B1=0;
	if(x & 0x100)CCP1CONbits.DC1B0=1; else CCP1CONbits.DC1B0=0;
	CCPR1L=(unsigned char)x;
}

void InitSPI(int speed)
{
	/* Initialise the SPI System */

	unsigned int i;	
	
	SDCS=1;
	SDCS_TRIS=0;	

	/* Configure the memory card sense pin */	

	powerSwitchComparator(0, 0);

	/* power cycle */
	
	if(T2CON==0)
	{
		/* Power is OFF, so turn ON */
		setMemoryCardPower(0);
		DelayMs(1);
		setMemoryCardPower(1);
		DelayMs(100);
	}
	
	#if(USE_PMDIS)
		PMDIS0bits.SPI2MD=0;
	#endif	
				
	if(speed!=SPI_SPEED)
	{
		/* slow */
		SSP2CON1=0x02;
	}
	else
	{
		/* fast */
		SSP2CON1=0x00;
	}	
	SSP2STAT=0x40;
	/* no Open Drain */
	ODCON3bits.SPI2OD=0;
	/* assumes that IOLOCK in PPSCON is cleared, as it is on power on reset */	
	TRISCbits.TRISC7=1;
	TRISBbits.TRISB0=0;
	TRISBbits.TRISB1=0;
	
	/* turn off analog functions */
	ANCON1bits.PCFG12=1;
	ANCON1bits.PCFG10=1;
	
	/* note that SCK2 needs to be set as both input and output to work properly */
	RPOR4=10;				// SDO2=10
	RPOR3=11;				// SCK2=11
	RPINR22=3;				// SCK2 input 
	RPINR21=SD_SDI_RPn;		// RPINR21= SDI2 function
	
	/* turn on the module */
	SSP2CON1bits.SSPEN=1;
	
	/* configure DMA (always needed even if SD not using it, as mem_cpy uses it) */
	
	DMACON1=0x08;			/* choose full duplex */
	SPI18SmallBuffer[0]=0xFF;
	
}

void closeSPI(void)
{
	SDCS=1;
	setMemoryCardPower(0);
	SSP2CON1=0;
	TRISCbits.TRISC7=1;
	TRISBbits.TRISB0=1;
	TRISBbits.TRISB1=1;
	RPOR4=0;
	RPOR3=0;
	RPINR22=0xFF;
	RPINR21=0xFF;
	#if(USE_PMDIS)
		PMDIS0bits.SPI2MD=1;
	#endif	
}

void powerSwitchComparator(unsigned char on, unsigned char edge)
{
	if(on)
	{
		#if(USE_PMDIS)
			PMDIS2bits.CMP3MD=0;
		#endif
		SDSENSE_TRIS=1;
		/* turn off pulls ups for port b */
		SDSENSEPULLUPS=1;
		if(edge)CM3CON=0x93; else CM3CON=0x8B;
		IPR5bits.CM3IP=0;
		PIR5bits.CM3IF=0;
		PIE5bits.CM3IE=1;
	}
	else
	{
		PIE5bits.CM3IE=0;
		#if(USE_PMDIS)
		//	PMDIS2bits.CMP3MD=1;
		#endif
		SDSENSE_TRIS=1;
		/* turn off pulls ups for port b */
		SDSENSEPULLUPS=1;
	}	
}

unsigned char WriteSPI(unsigned char x)
{
	SDCS=0;	
	SSP2BUF=x;
	while(SSP2STATbits.BF==0)Nop();
	x=SSP2BUF;
	SDCS=1;
	return x;
}

unsigned char WriteSPIWithoutSS(unsigned char x)
{
	SSP2BUF=x;
	while(SSP2STATbits.BF==0)Nop();
	x=SSP2BUF;
	return x;
}

unsigned char WriteSPIState(SPI_DESCRIPTOR* sd, unsigned char x)
{
	unsigned char ch;
	
	ch=((unsigned char)sd->ckdidocspin & 0x0F);
	/* ch is now the CS channel */
	setIOObject(ch, 0);
	SSP2BUF=x; 
	while(SSP2STATbits.BF==0)Nop();
	x=SSP2BUF;
	setIOObject(ch, 1);
	return x;
}

void ejectCard(void)
{
	closeSPI();
	memoryCardEjected=1;
	memoryCardSystemUp=0;
	cardInfo.cERROR=ERROR_CLOSED_CARD;
	cardInfo.SIZEFREE=0;
}

#if 1
void memoryCardTask(void)
{
	if((memoryCardSystemUp==0)&&(SDSENSE==SDSENSE_ON)&&(cardInfo.cERROR==ERROR_CLOSED_CARD))
	{
			initMemoryCardSystem(&cardInfo);
			if(memoryCardSystemUpMacro())
			{  	
				/* memory card good! */
				if(memoryCardEjected)
				{
					USBSoftDetach();
					syncSettingsTask();
					thisUDL.dataLogger.pors++;
					thisUDL.dataLogger.dirty|=DL_FILE_DIRTY;
					logHeader();
					USBDeviceAttach();
				}
			}
			else
			{
				/* bad memory card! */
			}
			memoryCardEjected=0;
	}	
	else
	if((memoryCardSystemUp==1)&&(SDSENSE!=SDSENSE_ON))
	{
			/* memory card ejected! */
			ejectCard();
	}
}	

void memoryCardEmptyPowerSaverTask(void)
{
	if((memoryCardSystemUp==0)&&(SDSENSE!=SDSENSE_ON))
	{
		if(thisUDL.dataLogger.ven.vmNum!=0)initVMEnvironment(&thisUDL.dataLogger.ven);
		if(hostAlive==0)
		{
			/* 
				Since SleepTime() calls memoryCardTask, we must ensure that this
				doesn't cause a stack overflow, by setting hostAlive to greater than 0! 
				before going into sleepTime()! (M.G.)
			*/
			hostAlive=HOST_ALIVE_ADDITIONAL_TIMEOUT;
			ejectCard();
			sleepTime(1);
		}
	}
}

#endif
	
void initSPIState(SPI_DESCRIPTOR* sd)
{
	unsigned char ch;
	
	/* turn off the module */
	SSP2CON1bits.SSPEN=0;
	
	if(sd->mode & SPI_NO_MODE)
	{
		ch=(unsigned char)(sd->ckdidocspin>>12) & 0x0F;
		/* ch is now the CK pin channel */
		setRPNValue(ch, 0);				/* SCK2=11 */
		RPINR22=0x1F;					/* SCK2 input */
		closeIOObject(ch);
		
		ch=(unsigned char)(sd->ckdidocspin>>8) & 0x0F;
		/* ch is now the DI pin channel */
		RPINR21=0x1F;					/* SDI2 function */
		closeIOObject(ch);
		
		ch=(unsigned char)(sd->ckdidocspin>>4) & 0x0F;
		/* ch is now the DO pin channel */
		setRPNValue(ch, 0); 			/* SDO2=10 */
		closeIOObject(ch);
	
		ch=(unsigned char)(sd->ckdidocspin) & 0x0F;
		/* ch is now the CS pin channel */
		closeIOObject(ch);
	}
	else
	{
	
	SSP2STAT=0x40;
	
	if(sd->mode & SPI_SPEED_DIV_64)
	{
		/* divide by 64 */
		SSP2CON1=0x02;
	}
	else
	if(sd->mode & SPI_SPEED_DIV_16)
	{
		/* divide by 16 */
		SSP2CON1=0x01;
	}
	else
	if(sd->mode & SPI_SPEED_DIV_8)
	{
		/* divide by 8 */
		SSP2CON1=0x0A;
	}
	else
	{
		/* fast, divide by 4 */
		SSP2CON1=0x00;
	}
	
	if(sd->mode & SPI_INV_CKE)
	{
		SSP2STATbits.CKE=0;
	}
	
	if(sd->mode & SPI_INV_CKP)
	{
		SSP2CON1bits.CKP=1;
	}

	if(sd->mode & SPI_OPEN_DRAIN)
	{
		ODCON3bits.SPI2OD=1;
	}
	else
	{
		ODCON3bits.SPI2OD=0;
	}
	
	if(sd->mode & SPI_SAMPLE_AT_END)
	{
		SSP2STATbits.SMP=1;
	}

	ch=(unsigned char)(sd->ckdidocspin>>12) & 0x0F;
	/* ch is now the CK pin channel */
	setRPNValue(ch, 11); 				/* SCK2=11 */
	RPINR22=getRPNNumber(ch);			/* SCK2 input */
	openIOObject(ch, 0);
		
	ch=(unsigned char)(sd->ckdidocspin>>8) & 0x0F;
	/* ch is now the DI pin channel */
	RPINR21=getRPNNumber(ch);			/* SDI2 function */
	openIOObject(ch, 1);
	
	ch=(unsigned char)(sd->ckdidocspin>>4) & 0x0F;
	/* ch is now the DO pin channel */
	setRPNValue(ch, 11); 				/* SDO2=10 */
	openIOObject(ch, 0);
	
	ch=(unsigned char)(sd->ckdidocspin) & 0x0F;
	/* ch is now the CS pin channel */
	setIOObject(ch, 1); 
	openIOObject(ch, 0);
	/* turn on the module */
	SSP2CON1bits.SSPEN=1;
	}
}

void closeSPIState(SPI_DESCRIPTOR* sd)
{
	unsigned char ch;
	
	ch=(unsigned char)(sd->ckdidocspin) & 0x0F;
	/* ch is now the CS pin channel */
	setIOObject(ch, 1);
	
	ch=(unsigned char)(sd->ckdidocspin>>12) & 0x0F;
	/* ch is now the CK pin channel */
	setRPNValue(ch, 0);				/* SCK2=11 */
	RPINR22=0x1F;					/* SCK2 input */
	closeIOObject(ch);
	
	ch=(unsigned char)(sd->ckdidocspin>>8) & 0x0F;
	/* ch is now the DI pin channel */
	RPINR21=0x1F;					/* SDI2 function */
	closeIOObject(ch);
	
	ch=(unsigned char)(sd->ckdidocspin>>4) & 0x0F;
	/* ch is now the DO pin channel */
	setRPNValue(ch, 0); 			/* SDO2=10 */
	closeIOObject(ch);
		
	/* Fast clock */
	SSP2CON1=0x00;
	SSP2STAT=0x40;
	/* note that SCK2 needs to be set as both input and output to work properly */
	RPOR4=10;				// SDO2=10
	RPOR3=11;				// SCK2=11
	RPINR22=3;				// SCK2 input 
	RPINR21=SD_SDI_RPn;		// RPINR21= SDI2 function
	/* no open drain */
	ODCON3bits.SPI2OD=0;
	/* turn on the module */
	SSP2CON1bits.SSPEN=1;
}

void reconnectMemoryCard(void)
{
	flushLog();
	saveSettingsFileTask();
	closeSPI();
	memoryCardEjected=1;
	memoryCardSystemUp=0;
	cardInfo.cERROR=ERROR_CLOSED_CARD;
	cardInfo.SIZEFREE=0;
}
